/*
 * Decompiled with CFR 0.152.
 */
package org.opensearch.bootstrap;

import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.Security;
import java.security.cert.CertificateException;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class FipsTrustStoreValidator {
    private static final Logger LOGGER = LogManager.getLogger(FipsTrustStoreValidator.class);
    private static final String TRUST_STORE_PATH_PROP = "javax.net.ssl.trustStore";
    private static final String TRUST_STORE_TYPE_PROP = "javax.net.ssl.trustStoreType";
    private static final String TRUST_STORE_PROVIDER_PROP = "javax.net.ssl.trustStoreProvider";
    private static final String TRUST_STORE_PASSWORD_PROP = "javax.net.ssl.trustStorePassword";
    private static final String ALLOWED_TYPE_PKCS11 = "PKCS11";
    private static final String ALLOWED_TYPE_BCFKS = "BCFKS";

    public static void validate() {
        FipsTrustStoreValidator.validate(System.getProperty(TRUST_STORE_PATH_PROP, ""), System.getProperty(TRUST_STORE_TYPE_PROP, ""), System.getProperty(TRUST_STORE_PROVIDER_PROP, ""), System.getProperty(TRUST_STORE_PASSWORD_PROP, ""));
    }

    protected static void validate(String trustStorePath, String trustStoreType, String trustStoreProvider, String trustStorePassword) {
        if (trustStoreType.isBlank()) {
            throw new IllegalStateException("Trust store type must be specified using the '-Djavax.net.ssl.trustStoreType' JVM option. Accepted values are PKCS11 and BCFKS.");
        }
        Map requiredProperties = switch (trustStoreType) {
            case ALLOWED_TYPE_PKCS11 -> Map.ofEntries(Map.entry(TRUST_STORE_TYPE_PROP, trustStoreType), Map.entry(TRUST_STORE_PROVIDER_PROP, trustStoreProvider));
            case ALLOWED_TYPE_BCFKS -> Map.ofEntries(Map.entry(TRUST_STORE_PATH_PROP, trustStorePath), Map.entry(TRUST_STORE_TYPE_PROP, trustStoreType), Map.entry(TRUST_STORE_PROVIDER_PROP, trustStoreProvider));
            default -> throw new IllegalStateException("Trust store type must be PKCS11 or BCFKS for FIPS compliance. Found: " + trustStoreType);
        };
        List<Map.Entry> missingProperties = requiredProperties.entrySet().stream().filter(entry -> ((String)entry.getValue()).isBlank()).toList();
        if (!missingProperties.isEmpty()) {
            throw new IllegalStateException("FIPS trust store is not set-up. Cannot find following JRE properties:\n" + missingProperties.stream().map(Map.Entry::getKey).collect(Collectors.joining("\n")));
        }
        if (ALLOWED_TYPE_BCFKS.equals(trustStoreType)) {
            FipsTrustStoreValidator.validateBCFKSFile(trustStorePath);
        }
        try {
            Provider provider = Security.getProvider(trustStoreProvider);
            if (provider == null) {
                throw new IllegalStateException("Trust store provider not available: " + trustStoreProvider);
            }
            KeyStore keyStore = KeyStore.getInstance(trustStoreType, provider);
            switch (trustStoreType) {
                case "PKCS11": {
                    keyStore.load(null, trustStorePassword.toCharArray());
                    break;
                }
                case "BCFKS": {
                    InputStream inputStream = Files.newInputStream(Path.of(trustStorePath, new String[0]), new OpenOption[0]);
                    try {
                        keyStore.load(inputStream, trustStorePassword.toCharArray());
                        if (inputStream == null) break;
                    }
                    catch (Throwable throwable) {
                        if (inputStream != null) {
                            try {
                                inputStream.close();
                            }
                            catch (Throwable throwable2) {
                                throwable.addSuppressed(throwable2);
                            }
                        }
                        throw throwable;
                    }
                    inputStream.close();
                    break;
                }
            }
            if (keyStore.size() == 0) {
                LOGGER.warn("Trust store is valid but contains no certificates (type: {})", (Object)trustStoreType);
            }
            LOGGER.debug("Trust store validation successful (type: {}, provider: {}, certificates: {})", (Object)trustStoreType, (Object)provider.getName(), (Object)keyStore.size());
        }
        catch (KeyStoreException e) {
            throw new IllegalStateException("Invalid trust store type or provider: " + e.getMessage(), e);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("Trust store algorithm not supported: " + e.getMessage(), e);
        }
        catch (CertificateException e) {
            throw new IllegalStateException("Trust store contains invalid certificates: " + e.getMessage(), e);
        }
        catch (IOException e) {
            throw new UncheckedIOException("Cannot read trust store file (possibly wrong password): " + e.getMessage(), e);
        }
        catch (Exception e) {
            throw new IllegalStateException("Trust store validation failed: " + e.getMessage(), e);
        }
    }

    private static void validateBCFKSFile(String trustStorePath) {
        Path trustStoreFile = Path.of(trustStorePath, new String[0]);
        if (!Files.exists(trustStoreFile, new LinkOption[0])) {
            throw new IllegalStateException("Trust store file does not exist: " + trustStorePath);
        }
        if (!Files.isReadable(trustStoreFile)) {
            throw new IllegalStateException("Trust store file is not readable: " + trustStorePath);
        }
        try {
            if (Files.size(trustStoreFile) == 0L) {
                throw new IllegalStateException("Trust store file is empty: " + trustStorePath);
            }
        }
        catch (IOException e) {
            throw new UncheckedIOException("Cannot access trust store file: " + e.getMessage(), e);
        }
    }
}

