﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/accessanalyzer/AccessAnalyzerRequest.h>
#include <aws/accessanalyzer/AccessAnalyzer_EXPORTS.h>
#include <aws/accessanalyzer/model/Criterion.h>
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>

#include <utility>

namespace Aws {
namespace AccessAnalyzer {
namespace Model {

/**
 * <p>Creates an archive rule.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/accessanalyzer-2019-11-01/CreateArchiveRuleRequest">AWS
 * API Reference</a></p>
 */
class CreateArchiveRuleRequest : public AccessAnalyzerRequest {
 public:
  AWS_ACCESSANALYZER_API CreateArchiveRuleRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateArchiveRule"; }

  AWS_ACCESSANALYZER_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name of the created analyzer.</p>
   */
  inline const Aws::String& GetAnalyzerName() const { return m_analyzerName; }
  inline bool AnalyzerNameHasBeenSet() const { return m_analyzerNameHasBeenSet; }
  template <typename AnalyzerNameT = Aws::String>
  void SetAnalyzerName(AnalyzerNameT&& value) {
    m_analyzerNameHasBeenSet = true;
    m_analyzerName = std::forward<AnalyzerNameT>(value);
  }
  template <typename AnalyzerNameT = Aws::String>
  CreateArchiveRuleRequest& WithAnalyzerName(AnalyzerNameT&& value) {
    SetAnalyzerName(std::forward<AnalyzerNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The name of the rule to create.</p>
   */
  inline const Aws::String& GetRuleName() const { return m_ruleName; }
  inline bool RuleNameHasBeenSet() const { return m_ruleNameHasBeenSet; }
  template <typename RuleNameT = Aws::String>
  void SetRuleName(RuleNameT&& value) {
    m_ruleNameHasBeenSet = true;
    m_ruleName = std::forward<RuleNameT>(value);
  }
  template <typename RuleNameT = Aws::String>
  CreateArchiveRuleRequest& WithRuleName(RuleNameT&& value) {
    SetRuleName(std::forward<RuleNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The criteria for the rule.</p>
   */
  inline const Aws::Map<Aws::String, Criterion>& GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  template <typename FilterT = Aws::Map<Aws::String, Criterion>>
  void SetFilter(FilterT&& value) {
    m_filterHasBeenSet = true;
    m_filter = std::forward<FilterT>(value);
  }
  template <typename FilterT = Aws::Map<Aws::String, Criterion>>
  CreateArchiveRuleRequest& WithFilter(FilterT&& value) {
    SetFilter(std::forward<FilterT>(value));
    return *this;
  }
  template <typename FilterKeyT = Aws::String, typename FilterValueT = Criterion>
  CreateArchiveRuleRequest& AddFilter(FilterKeyT&& key, FilterValueT&& value) {
    m_filterHasBeenSet = true;
    m_filter.emplace(std::forward<FilterKeyT>(key), std::forward<FilterValueT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A client token.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  CreateArchiveRuleRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_analyzerName;

  Aws::String m_ruleName;

  Aws::Map<Aws::String, Criterion> m_filter;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};
  bool m_analyzerNameHasBeenSet = false;
  bool m_ruleNameHasBeenSet = false;
  bool m_filterHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
};

}  // namespace Model
}  // namespace AccessAnalyzer
}  // namespace Aws
